#!/usr/bin/python
import math

from rgbkbd.core import StaticLighting, KeyboardMode
from rgbkbd.color import Color, Colors
from rgbkbd.geometry import Keys
from rgbkbd.animation import \
    AlternatingColorPattern, \
    BeatColorPattern, \
    ThrobColorPattern, \
    PulseColorPattern, \
    ColorAnimation

# Keyboard Modes
from rgbkbd.modes.basicmodes import StaticMode, RandomMode, ColorMode
from rgbkbd.modes.typingmode import TypingMode
from rgbkbd.modes.secretnotesmode import SecretNotesMode
from rgbkbd.modes.anglemotionmode import AngleMotionMode


class CommandMode(KeyboardMode):
    """Keyboard mode for selecting a keyboard mode"""
    _color_modes = [
        ("grave", Colors.BLACK),
        ("1", Colors.DARK_GREY),
        ("2", Colors.GREY),
        ("3", Colors.WHITE),
        ("4", Colors.RED),
        ("5", Colors.PURPLE),
        ("6", Colors.BLUE),
        ("7", Colors.CYAN),
        ("8", Colors.GREEN),
        ("9", Colors.YELLOW),
        ("0", Colors.ORANGE),
    ]
    _color_mode_keys = [k for k, _c in _color_modes]
    _color_mode_by_key = dict(_color_modes)

    def __init__(self, manager, keyboard):
        static_key_lighting = [
            (Keys.ALL, Colors.BLACK), # black out everything not otherwise set
            ("light,lock", Colors.WHITE), # exit command mode
            ("scroll", Colors.GREEN), # discrete writing mode

            ("prtscn", Colors.WHITE), # typing mode
        ] + self._color_modes

        static = StaticLighting(keyboard, profile=static_key_lighting)
        animations = [
            # Key for random pattern of lit keys
            ColorAnimation(keys="home", keyboard=keyboard,
                color_pattern=AlternatingColorPattern(colors=[Color.random8() for n in range(50)], period=0.8*50)),

            # Animation control panel
            ColorAnimation(keys="stop", keyboard=keyboard,
                color_pattern=AlternatingColorPattern(colors=[Colors.WHITE, Colors.BLACK], period=3)),
            ColorAnimation(keys="prev", keyboard=keyboard,
                color_pattern=PulseColorPattern(colors=[Colors.WHITE, Colors.BLACK], period=3)),
            ColorAnimation(keys="play", keyboard=keyboard,
                color_pattern=ThrobColorPattern(colors=[Colors.WHITE, Colors.BLACK], period=3)),
            ColorAnimation(keys="next", keyboard=keyboard,
                color_pattern=BeatColorPattern(colors=[Colors.WHITE, Colors.BLACK], period=3)),

            # Direction control
            ColorAnimation(keys="num5", keyboard=keyboard,
                color_pattern=ThrobColorPattern(colors=[Colors.BLACK, Colors.WHITE], period=3)),
            ColorAnimation(keys="num1,num2,num3,num4,num6,num7,num8,num9",
                keyboard=keyboard,
                color_pattern=ThrobColorPattern(colors=[Colors.WHITE, Colors.BLACK], period=3)),
        ]
        super(CommandMode, self).__init__(manager, keyboard, static, animations)
        self.previous_mode = None

    def start(self, previous_mode=None):
        """Initialize the keyboard for command mode"""
        #print "Starting command mode" # DEBUG
        super(CommandMode, self).start()
        self.previous_mode = previous_mode
        self.keyboard.unbind()

    def chord_event(self, chord):
        """Handle chords as commands"""
        if chord == ["light"]:
            # Allow turning off the lights
            self.lights_on = not self.lights_on
            self.keyboard.lights(self.lights_on)
        elif sorted(chord) == ['light', 'lock']:
            # exit command mode back to the previous mode
            #print "Exiting command mode" # DEBUG
            self.keyboard.rebind()
            self.manager.mode_start(self.previous_mode)
        # There are a few keyboard modes that don't take or need color modifiers
        elif chord == ["scroll"]: # SecretNotes mode
            self.manager.mode_start(SecretNotesMode(manager=self.manager, keyboard=self.keyboard))
        elif chord == ["prtscn"]: # Typing speed mode
            self.manager.mode_start(TypingMode(manager=self.manager, keyboard=self.keyboard))
        elif chord == ["home"]: # Random mode with random color selection
            self.manager.mode_start(RandomMode(manager=self.manager, keyboard=self.keyboard))
        # Modes that support color modifiers
        elif set(self._color_mode_keys).intersection(chord):
            # They hit at least one of the color mode keys
            directions = {
                "num6": (0.0  * math.pi, 50),
                "num9": (0.25 * math.pi, 30),
                "num8": (0.5  * math.pi, 10),
                "num7": (0.75 * math.pi, 30),
                "num4": (1.0  * math.pi, 50),
                "num1": (1.25 * math.pi, 30),
                "num2": (1.5  * math.pi, 10),
                "num3": (1.75 * math.pi, 30),
                "num5": (None, None), # No motion; might want it to be a circular pattern?
            }
            mode_types = {
                "home": RandomMode,
            }
            color_pattern_types = {
                "stop": AlternatingColorPattern,
                "prev": PulseColorPattern,
                "play": ThrobColorPattern,
                "next": BeatColorPattern,
            }
            direction = None
            mode_type = StaticMode
            colors = []
            color_pattern = None
            for key in chord:
                if key in mode_types.keys():
                    mode_type = mode_types[key]
                elif key in self._color_mode_keys:
                    colors.append(self._color_mode_by_key[key])
                elif key in directions.keys():
                    # Just take the last direction
                    direction, size = directions[key]
                elif key in color_pattern_types.keys():
                    # Just take the last pattern
                    color_pattern = color_pattern_types[key]
                else:
                    # ERROR: an unrecognized key was pressed
                    return
            foreground = colors[0]
            if len(colors) > 1:
                background = colors[1]
            else:
                colors.append(Colors.BLACK)
                background = Colors.BLACK

            self.keyboard.rebind()

            if color_pattern is None:
                #print "No color pattern"
                mode = mode_type(manager=self.manager, keyboard=self.keyboard,
                    foreground=foreground, background=background)
            else:
                if direction is None:
                    #print "No direction" # DEBUG
                    mode = ColorMode(manager=self.manager,
                        keyboard=self.keyboard, colors=colors, period=15,
                        color_pattern_cls=color_pattern)
                else:
                    #print "Direction is %s" % direction # DEBUG
                    mode = AngleMotionMode(manager=self.manager,
                        keyboard=self.keyboard, colors=colors, period=15,
                        color_pattern_cls=color_pattern, angle=direction, size=size*len(colors)/2)

            self.manager.mode_start(mode)
        # Ignore unrecognized chord events
