#!/usr/bin/env python
"""
Syntax: MergeActor.py ticketnum component version requestor

Verify that a branch can be merged to its trunk without conflicts, but don't
commit the merge.
"""

import os
import sys
import trac.env

import SvnOps
from WorkQueue import MergeBotActor, VersionToDir
from TrackerTools import GetWorkDir, GetRepositoryLocalUrl, Task, GetLogFile

def check_merge_action(trac_env, ticketnum, component, version, requestor):
    """
    Verify that a branch can be merged to its trunk without conflicts, but
    don't commit the merge.
    """
    task_obj = Task(trac_env, ticketnum)
    endstatus = "???????"
    workdir = GetWorkDir(trac_env, ticketnum, __name__)
    logfile = GetLogFile(trac_env, ticketnum)
    # FIXME: Should we just bail out instead?
    if os.path.exists(workdir):
        os.system("rm -rf \"%s\"" % (workdir))

    sourceurl = os.path.join(GetRepositoryLocalUrl(trac_env), component,
        VersionToDir(version))
    ticketurl = os.path.join(GetRepositoryLocalUrl(trac_env), component, "branches",
        "ticket-%s" % ticketnum)

    branch_info = SvnOps.get_branch_info(ticketurl, GetLogFile(trac_env, ticketnum))
    # FIXME: if not branch_info: # Error case
    startrev, endrev = branch_info

    SvnOps.checkout(sourceurl, workdir, logfile)
    # TODO: check return code of the above
    results = SvnOps.merge(ticketurl, workdir, (startrev, endrev), logfile)
    conflicts = SvnOps.conflicts_from_merge_results(results)
    if conflicts:
        message = "\n".join([
            "Found %s conflicts while checking merge of %s:%s to %s for %s." % \
                (len(conflicts), startrev, endrev, version, requestor),
            "Files in conflict:",
            "{{{",
            "\n".join(conflicts),
            "}}}",
            "A rebranch will be needed before this can be merged.",
        ])
        endstatus = "conflicts"
    else:
        message = \
            "Found no conflicts while checking merge of %s:%s to %s for %s." \
            % (startrev, endrev, version, requestor)
        endstatus = "branched"

    # Clean up the work area
    os.system("rm -rf \"%s\"" % (workdir, ))

    task_obj.AddComment(message)
    return endstatus, task_obj

class CheckMergeActor(MergeBotActor):
    "Actor wrapper for the check_merge_action"
    def __init__(self, trac_env):
        MergeBotActor.__init__(self, trac_env, "checkmerge", check_merge_action)

def main():
    tracdir = sys.argv[1]
    trac_env = trac.env.open_environment(tracdir)
    mergingActor = CheckMergeActor(trac_env)
    mergingActor.AddTask(sys.argv[1:])
    mergingActor.Run()

if __name__ == "__main__":
    main()

# vim:foldcolumn=4 foldmethod=indent
# vim:tabstop=4 shiftwidth=4 softtabstop=4 expandtab
