"""Base class for mergebot actors that do the various kinds of tasks
"""
import os
from mergebot.svn import SvnLib
SvnOps = SvnLib()

class Actor(object):
    """Base class for mergebot actors"""
    def __init__(self, work_dir, repo_url, repo_dir, ticket, component,
                 version, summary, requestor):
        self.work_dir = work_dir
        self.repo_url = repo_url
        self.repo_dir = repo_dir
        self.ticket = ticket
        self.component = component
        self.version = version
        self.summary = summary
        self.requestor = requestor

    def execute(self):
        """To be overridden by subclasses.
        Returns a tuple containing:
        a dictionary of fields to update in the ticket
        a comment string
        True on success, False on failure
        """
        raise NotImplementedError

    def logfilename(self):
        """Returns the absolute path of the logfile for this ticket"""
        return os.path.abspath(os.path.join(os.path.dirname(self.work_dir),
            'ticket-%s.log' % self.ticket))

    def public_url(self):
        """Returns the public URL for this component"""
        return '%s/%s' % (self.repo_url, self.component)

    def branch_public_url(self):
        """Returns the public svn url of the branch for this ticket.
        """
        return '%s/branches/ticket-%s' % (self.public_url(), self.ticket)

    def local_url(self):
        """Returns the local URL for this component"""
        return 'file://%s/%s' % (self.repo_dir, self.component)

    def baseline_local_url(self):
        """Returns the local svn url this ticket branches from.
        """
        return '%s/%s' % (self.local_url(), self.version_subdir())

    def branch_local_url(self):
        """Returns the local svn url of the branch for this ticket.
        """
        return '%s/branches/ticket-%s' % (self.local_url(), self.ticket)

    def version_subdir(self):
        """Returns the subdirectory name for the version"""
        if self.version == 'trunk':
            subdir = 'trunk'
        elif self.version.startswith('#'):
            # branched from another ticket
            subdir = 'branches/ticket-%s' % self.version.lstrip('#')
        else:
            # assume release branch
            subdir = 'branches/release-%s' % self.version
        return subdir

    def check_required_directories(self):
        """Make sure the various urls we require do exist"""
        if not SvnOps.does_url_exist(self.local_url()):
            return 'Component %s does not exist in the repository.' \
                % self.component
        if not SvnOps.does_url_exist(self.local_url() + '/branches'):
            return 'No directory in which to create branches for ' \
                'component %s in the repository.' % self.component
        if not SvnOps.does_url_exist(self.baseline_local_url()):
            return 'Version %s for component %s does not exist in the ' \
                'repository.' % (self.version, self.component)
        return None

